import os
import sys
import math
import functions

# Task: For each outer stem, perform MWIS calculation with confidence as weights
# on the set of all possible inner stems.
# This returns bulge loops, internal loops and multiloops.
def internal_mwis(matrix_stems_ib):
    structures_dic, candidate_list = {}, []    
    for stem, values in matrix_stems_ib.items():        
        entry = (stem[0], stem[1], values[0], values[1], values[1])
        candidate_list.append(entry)
    candidate_list.sort()    
    sorted_endpoint_list = functions.create_sorted_endpointlist(candidate_list)
    for endpoint in sorted_endpoint_list:   # Scan sorted endpoints list                
        if endpoint[1] == 'r':              # If a right endpoint is scanned
            sorted_endpoint_list_recursive, nested = [], []
            index = endpoint[3]
            nested = find_nested(endpoint,candidate_list)
            if nested:                      # MWIS on the set of inner stems
                sorted_endpoint_list_recursive = functions.create_sorted_endpointlist(nested)                
                result = functions.MWIS(nested, sorted_endpoint_list_recursive)
                interval = candidate_list[index-1]
                confidence = interval[4]
                for element in result:    # Confidence sum
                    confidence = confidence + element[4]
                # Store updated confidence for outer stem
                candidate_list[index-1] = (interval[0],interval[1],interval[2],interval[3],confidence)
                stem = interval[0], interval[1], interval[2]
                # Store inner structure elements in dictionary
                structures_dic[stem] = result        
            else:
                interval = candidate_list[index-1]
                stem = interval[0],interval[1],interval[2]
                structures_dic[stem] = []
    return structures_dic

# Task: Function for finding all inner stems, given a right endpoint of an outer stem
# Flexible inner stem finding, allow certain base pair overlap
def find_nested(endpoint,candidate_list):
    result = []
    interval_index = endpoint[3]
    interval = candidate_list[interval_index-1]
    interval_left = interval[0] + interval[2]
    interval_right = interval[1] - interval[2]
    stemlength_outer = interval[2]    
    for compare_interval in candidate_list:
        stemlength_inner = compare_interval[2]
        if compare_interval[0] >= interval_left:                              
            if compare_interval[1] <= interval_right:
                result.append(compare_interval)
        if stemlength_inner > 3 and stemlength_outer > 3:   # Overlap of 1 bp
            if compare_interval[0] >= interval_left - 1:                                                          
                if compare_interval[1] <= interval_right + 1:
                    result.append(compare_interval)                                            
        if stemlength_inner > 4 and stemlength_outer > 4:   # Overlap of 2 bp
            if compare_interval[0] >= interval_left - 2:                                                           
                if compare_interval[1] <= interval_right + 2:
                    result.append(compare_interval)
    result = list(set(result))          # Remove duplicates from list
    result.sort()
    return result

# Task: Assembling secondary structure elements recursively 
# from dictionary to produce dot-bracket notation
def print_recursive(structures_dic,stem_list_recursive,stem_internal,ml):
    stem_internal_list = structures_dic[stem_internal[0],stem_internal[1],stem_internal[2]]
    if stem_internal_list:
        if len(stem_internal_list) > 1:
            ml = 1
        for stem in stem_internal_list:            
            stem_list_recursive.append(stem)
            stem_list_recursive, ml = print_recursive(structures_dic,stem_list_recursive,stem,ml)            
    return stem_list_recursive, ml

# Task: Function for evaluation of local stem energies
# Store secondary structure elements with structure in file for RNAeval
# Update dictionaries with both stacking and free energy weights
def evaluation_bulge_internal(structures_dic, seq):    
    bulges_internal = file("bulge_internal_structures.txt",'w')
    multiloops = file("multiloop_structures.txt",'w')    
    for stem in sorted(set(structures_dic)):
        ml, stem_list_recursive, string_list = 0, [], []
        start, end, length_stem = stem[0], stem[1], stem[2]
        string = "> " + str(start) + " " + str(end)+ " " + str(length_stem) + "\n"
        local_sequence = seq[start-1:end]        
        for i in xrange(len(local_sequence)):
            string_list.append('.')
        for i in xrange(length_stem):    
            string_list[i] = '('
            string_list[len(local_sequence)-i-1] = ')'            
        nested = structures_dic[stem]        
        if nested:            
            if len(nested) > 1:
                ml = 1            
            for stem_internal in nested:    # Look up all internal structure elements
                stem_list_recursive.append(stem_internal)
                nested_recursive = structures_dic[stem_internal[0],stem_internal[1],stem_internal[2]]
                if len(nested_recursive) > 1:
                    ml = 1
                stem_list_recursive, ml = print_recursive(structures_dic,stem_list_recursive,stem_internal,ml)                               
            stem_list_recursive.sort()
            # Insert recursive stems in dot-bracket notation
            for stem in stem_list_recursive:
                stem_start = stem[0]
                stem_end = stem[1]
                length_stem = stem[2]
                for i in xrange(length_stem):
                    if string_list[stem_start-start+i] != '(':
                        if string_list[stem_start-start+i] != ')':                            
                            if string_list[stem_end - start - i] != ')':
                                if string_list[stem_end - start - i] != '(':
                                    string_list[stem_start-start+i] = '('
                                    string_list[stem_end - start - i] = ')'               
            if start == 1:
                if end != len(seq):     # Dangling end on the right only
                    local_sequence = seq[start-1:end+1]
                    string_list.append(":")     
                else:                   # No dangling ends
                    local_sequence = seq[start-1:end]
            else:
                if end != len(seq):     # Dangling end on both ends
                    local_sequence = seq[start-2:end+1]
                    string_list.insert(0,":")
                    string_list.append(":")         
                else:                   # Dangling end on the left only
                    local_sequence = seq[start-2:end]
                    string_list.insert(0,":")                    
            s = "".join(string_list)
            
            if ml == 1:                 # If it is a multiloop                
                multiloops.write(string)                                        
                multiloops.write(str(local_sequence))
                multiloops.write('\n')                     
                multiloops.write(str(s))
                multiloops.write('\n')                                                                                                               
            if ml == 0:                 # If it is not a multiloop
                bulges_internal.write(string)                               
                bulges_internal.write(str(local_sequence))
                bulges_internal.write('\n')                     
                bulges_internal.write(str(s))
                bulges_internal.write('\n')                                           
    bulges_internal.close()
    multiloops.close()   
    # Evaluate free energies for each structure element
    eval_loops = os.popen("cat multiloop_structures.txt | ViennaRNA-1.8.4/Progs/RNAeval") 
    energy_loops = eval_loops.read()        
    output = file("multiloops_energy.txt",'w')
    output.write(energy_loops)    
    output.close()        
    eval_loops = os.popen("cat bulge_internal_structures.txt | ViennaRNA-1.8.4/Progs/RNAeval") 
    energy_loops = eval_loops.read()            
    output = file("bulge_internal_energy.txt",'w')
    output.write(energy_loops)    
    output.close()
    
    stem_loops_list = []
    output = file("bulge_internal_energy.txt",'r')
    for i in output:
        stem_loops_list.append(i)
    output.close()    
    bulge_internal_dic = {}     # Bulge and internal loop dictionary 
    for i in xrange(0,len(stem_loops_list),3):
        info_stack = stem_loops_list[i+2].split()
        dot_bracket = info_stack[0]        
        if len(info_stack) > 2:
            free_energy = info_stack[2]
            free_energy = float(str(free_energy)[0:len(free_energy)-1])      
        else:
            free_energy = info_stack[1]
            free_energy = float(str(free_energy)[1:len(free_energy)-1])            
        info = stem_loops_list[i].split()
        key = int(info[1]), int(info[2])
        length = int(info[3])
        bulge_internal_dic[key] = length, dot_bracket, free_energy

    stem_loops_list = []
    output = file("multiloops_energy.txt",'r')
    for i in output:
        stem_loops_list.append(i)
    output.close()    
    multiloops = {}         # Multiloop dictionary
    for i in range(0,len(stem_loops_list),3):
        info_stack = stem_loops_list[i+2].split()
        dot_bracket = info_stack[0]        
        if len(info_stack) > 2:
            free_energy = info_stack[2]
            free_energy = float(str(free_energy)[0:len(free_energy)-1])      
        else:
            free_energy = info_stack[1]
            free_energy = float(str(free_energy)[1:len(free_energy)-1])            
        info = stem_loops_list[i].split()
        key = int(info[1]), int(info[2])
        length = int(info[3])
        multiloops[key] = length, dot_bracket, free_energy
                                
    return bulge_internal_dic, multiloops

# Task: Filtering of structure elements with high free energy
def filter_stems(bulge_internal_dic, cutoff_loops):    
    for stem in sorted(set(bulge_internal_dic)):
        if bulge_internal_dic[stem][2] >= cutoff_loops:
            del bulge_internal_dic[stem]
    return bulge_internal_dic    
